<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Mission;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ProgressController extends Controller
{
    public function getProfile(Request $request)
    {
        $user = $request->user()->load(['badgesEarned', 'missionsCompleted']);
        return response()->json($user);
    }

    public function startMission(Request $request, Mission $mission)
    {
        $user = Auth::user();

        $existingRecord = $user->missionsCompleted()->where('mission_id', $mission->id)->first();

        if ($existingRecord) {
            if ($existingRecord->pivot->status == 'completed') {
                return response()->json([
                    'message' => 'Misi ini sudah pernah Anda selesaikan sebelumnya.',
                    'status' => 'completed'
                ], 200);
            }
            return response()->json([
                'message' => 'Misi sedang berlangsung.',
                'status' => 'in_progress'
            ], 200);
        } else {
            $user->missionsCompleted()->attach($mission->id, [
                'status' => 'in_progress'
            ]);
            return response()->json([
                'message' => 'Misi dimulai!',
                'status' => 'in_progress'
            ], 201);
        }
    }

    public function finishMission(Request $request, Mission $mission)
    {
        $user = Auth::user();

        $userMissionPivot = $user->missionsCompleted()
                                 ->where('mission_id', $mission->id)
                                 ->first()?->pivot;

        if (!$userMissionPivot) {
             return response()->json(['message' => 'Misi belum dimulai. Silakan ambil misi di menu.'], 400);
        }

        if ($userMissionPivot->status === 'completed') {
            return response()->json([
                'message' => 'Misi ini sudah selesai!',
                'points_awarded' => 0,
                'total_points' => $user->total_points,
                'is_level_up' => false,
                'current_level' => $user->level ?? 1
            ], 200);
        }

        $pointsBefore = $user->total_points;
        $levelBefore = $user->level ?? 1;

        $user->total_points += $mission->points_reward;

        $newLevel = User::calculateLevel($user->total_points);
        $isLevelUp = $newLevel > $levelBefore;

        if ($isLevelUp) {
            $user->level = $newLevel;
        }

        if ($mission->badge_reward_id) {
            $hasBadge = $user->badgesEarned()
                             ->where('badge_id', $mission->badge_reward_id)
                             ->exists();
            if (!$hasBadge) {
                $user->badgesEarned()->attach($mission->badge_reward_id);
            }
        }

        $user->save();

        $user->missionsCompleted()->updateExistingPivot($mission->id, [
            'status' => 'completed',
            'completed_at' => now()
        ]);

        $nextMission = Mission::where('id', '>', $mission->id)
                              ->orderBy('id', 'asc')
                              ->first();

        $responseData = [
            'message' => 'Selamat! Misi berhasil diselesaikan.',
            'points_awarded' => $mission->points_reward,
            'total_points' => $user->total_points,
            'is_level_up' => $isLevelUp,
            'current_level' => $newLevel,
        ];

        if ($nextMission) {
            $responseData['next_mission'] = [
                'id' => $nextMission->id,
                'title' => $nextMission->title,
            ];
        } else {
            $responseData['next_mission'] = null;
        }

        return response()->json($responseData);
    }
}